// Version/Precision tags are added programmatically on shader load

uniform vec4 uClipPlane;
uniform vec3 uLightAmb;
uniform vec3 uLightDif;
uniform vec3 uLightSky;
uniform vec3 uFogColor;
uniform vec3 uFowColor;

uniform sampler2D uFowTexture;
//uniform sampler2D uShadowTexture;
uniform sampler2DShadow uShadowTexture; // sampler2DShadow requires GL_TEXTURE_COMPARE_MODE = GL_COMPARE_R_TO_TEXTURE

varying vec3 C;
varying vec3 L;
varying vec3 N;
varying vec3 P;

varying float fogValue;
varying vec3 ShadowCoord;

// Used by Shadows and FOW
// Using "uniform" could be slow, using "const" instead does not work on some GPUs (needs version 150 ontop?)
// There are ways to add this into main as consts and etc. Google "glsl poisson"
uniform vec2 poissonDisk[16] = vec2[16](
  vec2(-0.94201624, -0.39906216),
  vec2( 0.94558609, -0.76890725),
  vec2(-0.09418410, -0.92938870),
  vec2( 0.34495938,  0.29387760),
  vec2(-0.91588581,  0.45771432),
  vec2(-0.81544232, -0.87912464),
  vec2(-0.38277543,  0.27676845),
  vec2( 0.97484398,  0.75648379),
  vec2( 0.44323325, -0.97511554),
  vec2( 0.53742981, -0.47373420),
  vec2(-0.26496911, -0.41893023),
  vec2( 0.79197514,  0.19090188),
  vec2(-0.24188840,  0.99706507),
  vec2(-0.81409955,  0.91437590),
  vec2( 0.19984126,  0.78641367),
  vec2( 0.14383161, -0.14100790)
);

float computeShadow(vec3 aNormal) {
  #ifdef SHADOWS
    #define SHADOW_BIAS %
    #define SHADOW_SAMPLES %
    #define SHADOW_SAMPLING_AREA %
    
    // sampler2D
    /*
    float bias = clamp(0.005 * tan(acos(clamp(dot(aNormal, L), 0.0, 1.0))), 0, 0.01);
    float visibility = 1.0;
    if (texture(uShadowTexture, ShadowCoord.xy).z < ShadowCoord.z - bias) {
     visibility = 0.0;
    }
    */
  
    // sampler2DShadow
    /*
    float bias = clamp(0.005 * tan(acos(max(0.0, dot(aNormal, L)))), 0, 0.01);
    float visibility = texture(uShadowTexture, vec3(ShadowCoord.xy, (ShadowCoord.z - bias) / ShadowCoord.w));
    */

    // sampler2D with Poisson Sampling
    /*
    float bias = clamp(0.005 * tan(acos(clamp(dot(aNormal, L), 0.0, 1.0))), 0, 0.01);
    float visibility = 1.0;
    for (int i = 0; i < SHADOW_SAMPLES; i++){
      if (texture(uShadowTexture, ShadowCoord.xy + poissonDisk[i] * SHADOW_SAMPLING_AREA).z < ShadowCoord.z - bias) { 
        visibility -= 1.0 / SHADOW_SAMPLES;
      }
    }
    */

    // sampler2D with Poisson Sampling and Distance blur (does not look good)
    /*  
    float bias = clamp(0.005 * tan(acos(clamp(dot(aNormal, L), 0.0, 1.0))), 0, 0.01);
    float visibility = 1.0;
    float dst = clamp(ShadowCoord.z - texture(uShadowTexture, ShadowCoord.xy).z, 0.0, 1.0) * 10.0 * SHADOW_SAMPLING_AREA;
    for (int i = 0; i < SHADOW_SAMPLES; i++){
      if (texture(uShadowTexture, ShadowCoord.xy + poissonDisk[i] * dst).z < ShadowCoord.z - bias) { 
        visibility -= 1.0 / SHADOW_SAMPLES;
      }
    }
    */
 
    // sampler2DShadow with Poisson Sampling
    float bias = clamp(tan(acos(max(0.0, dot(aNormal, L)))), 0.2, 2.0) * SHADOW_BIAS;
    float res = 0.0;
    for (int i = 0; i < SHADOW_SAMPLES; i++){
      float smp = texture(uShadowTexture, vec3(ShadowCoord.xy + poissonDisk[i] * SHADOW_SAMPLING_AREA, ShadowCoord.z - bias));
      res += smp / SHADOW_SAMPLES;
    }
    
    return res;
  #else
    return 1.0;
  #endif
}

void main()
{
  /* Discard pixels behind Clipping Plane */
  float clipPos = dot(P, uClipPlane.xyz) + uClipPlane.w;
  if (clipPos < 0.0) {
    discard;
  }

  /* FOW */
  #define FOW_SAMPLES %
  #define FOW_SAMPLING_AREA %
  #define FOW_TEXTURE_SCALE %

  // Sample FOW with 0.5 offset because it is vertice-based, not tile/pixel based
  vec2 uv = (P.xy + vec2(0.5, 0.5)) * vec2(FOW_TEXTURE_SCALE, FOW_TEXTURE_SCALE);

  // Rough FOW
  //float fow = texture2D(uFowTexture, uv).r;
  
  // Blurred FOW looks much better
  /* Terrain is especially revealing of FOW imperfections, hence we prettify terrain FOW by bluring it */
  float fow = texture(uFowTexture, uv).r * 0.5; // Half the weight comes from base sampling
    for (int i = 0; i < FOW_SAMPLES; i++) {
      float fg = texture(uFowTexture, uv + poissonDisk[i] * FOW_SAMPLING_AREA).r;
      fow += (fg / FOW_SAMPLES) * 0.5; // blur samples have 0.5 weight
    }

  if (fow <= 0.0) {
    discard;
  }

  vec3 nN = normalize(N);

  float visibility = computeShadow(nN);

  float dotDirection = clamp(dot(nN, L), 0.0, visibility);
  vec3 diffuse = dotDirection * uLightDif;
  vec3 light = diffuse + uLightAmb + uLightSky;

  vec3 color = mix(C.rgb * light.rgb, uFogColor, fogValue);
  color = mix(uFowColor, color, fow);

  gl_FragColor = vec4(color, 1.0);
}